//************************************************************************
// T6963.C
// LowLevel Routinen fr LCD-Displays mit T6963
// Getestet mit AVR/ATMega bis 16 MHz.
//
// Der Nullpunkt des Displays liegt oben,links. Fr unten rechts
// mssen dann zustzliche Umrechnungen stattfinden.
//
// Mit SetPixel hat man den Grundstein fr alle
// erdenklichen Zeichenroutinen:
// Linien, Rechtecke, Kreise, Fllen, Muster
//
// hk@holger-klabunde.de
// http://www.holger-klabunde.de
// 05.03.2006
// Compiler AVR-GCC 3.4.3
//************************************************************************

#include <avr/io.h>

#include "protos.h"
#include "T6963.h"
#include "draw.h"

//##########################################################
void CheckBusy(void)
//##########################################################
{
 unsigned char by;

 DATA_DIR_IN(); //DatenPort auf Eingang zum lesen
 CTRL();    //Status Register lesen

 do         //Schleife Status Bits checken
  {
   CE_OFF();
   RD_OFF();
   NOP;    //Ein Waitstate fr 8 MHZ
   NOP;    //plus noch einen fr 16 MHZ
   by=READ_DATA(); //Status lesen
   RD_ON();
   CE_ON();
  }while((by&3) != 3); //Solange STA0 oder STA1 = 0

 DATA();   //Zurck auf DisplayDaten schreiben
}

//##########################################################
void WriteCTRL(unsigned char dat)
//##########################################################
{
 DATA_DIR_OUT(); //DatenPort auf Ausgang zum schreiben
 CTRL();     //Commando schreiben vorbereiten
 WRITE_DATA(dat); //Daten zum Commando

 CE_OFF();
 WR_OFF();
 NOP;    //Ein Waitstate fr 8 MHZ
 NOP;    //plus noch einen fr 16 MHZ
 WR_ON();
 CE_ON();

 DATA();   //Zurck auf DisplayDaten schreiben
}

//##########################################################
//Schreibt ein Datenbyte und ein Kommandobyte
void WriteData1(unsigned char dat, unsigned char command)
{
 WriteData(dat); //Daten schreiben
 WriteCommand(command); //Kommando schreiben
}

//##########################################################
//Schreibt zwei Datenbytes und ein Kommandobyte
void WriteData2(unsigned int dat, unsigned char command)
{
 WriteData((unsigned char)(dat&0xFF)); //Daten schreiben
 WriteData((unsigned char)(dat>>8)); //Daten schreiben
 WriteCommand(command); //Kommando schreiben
}

//##########################################################
//Datenbyte schreiben
void WriteData(unsigned char dat)
{
 CheckBusy(); //StatusBits STA0,1 checken

 DATA_DIR_OUT(); //DatenPort auf Ausgang zum schreiben
 DATA();     //DisplayDaten schreiben vorbereiten
 WRITE_DATA(dat); //Daten auf den Port

 CE_OFF();
 WR_OFF();
 NOP;    //Ein Waitstate fr 8 MHZ
 NOP;    //plus noch einen fr 16 MHZ
 WR_ON();
 CE_ON();
}

//##########################################################
//Kommandobyte schreiben
void WriteCommand(unsigned char command)
{
 CheckBusy(); //StatusBits STA0,1 checken
 WriteCTRL(command); //Kommando schreiben
}

//##########################################################
//Datenbyte lesen
unsigned char ReadData(void)
{
 unsigned char by;
 
 CheckBusy(); //StatusBits STA0,1 checken

 DATA_DIR_IN(); //DatenPort auf Eingang zum lesen
 DATA();     //DisplayDaten lesen vorbereiten

 CE_OFF();
 RD_OFF();
 NOP;    //Ein Waitstate fr 8 MHZ
 NOP;    //plus noch einen fr 16 MHZ
 by=READ_DATA(); //Daten vom Port lesen
 RD_ON();
 CE_ON();
 return by;
}

//##########################################################
//Pixel auf dem Display setzen
//mode=0; Pixel lschen
//mode=1; Pixel setzen
void SetPixel(unsigned char xpos, unsigned char ypos, unsigned char mode)
{
 unsigned int adress;
 unsigned char tmp,pixel;

 if(xpos>=LCD_WIDTH) return; //Punkt auerhalb der Anzeige

// if(ypos>=LCD_HEIGHT) return;

//Position im Display-RAM setzen
 adress=G_BASE; //Start of graphicarea

//anhand von xpos und ypos ermitteln welches Byte
//gelesen/geschrieben werden soll
 adress+= BYTES_PER_ROW * ypos;
 adress+= xpos / PIXPERBYTE;

 if(adress>G_END1) return; //Punkt auerhalb Display RAM

//Set address pointer
 WriteData((unsigned char)(adress&0xFF)); //Daten schreiben
 WriteData((unsigned char)(adress>>8)); //Daten schreiben
 WriteCommand(0x24); //Kommando schreiben

 if(mode==0) pixel=0xF0; //Kommando Pixel lschen
 else pixel=0xF8;        //Kommando Pixel setzen

//Die Reihenfolge der Pixel ist im Byte genau umgekehrt !
 tmp=(PIXPERBYTE-1)-(xpos%PIXPERBYTE);
 pixel|=tmp; //Position des Pixels im Byte setzen

 WriteCommand(pixel); //Befehl mit gewnschtem Pixel schreiben
}

//##########################################################
void DisplayOn()
//##########################################################
{
 INIT_CONTROL_PINS(); //set WR,CE,RD,C/D to outputs
 
 WR_ON();
 CE_ON();
 RD_ON();
 DATA();

 DATA_DIR_OUT(); //DatenPort auf Ausgang zum schreiben
 
//Write text home address=0x0000
 WriteData2(T_BASE,0x40);

//Write graphic home address
 WriteData2(G_BASE,0x42);

//Text area column set= Setze Textspalten 
 WriteData2(BYTES_PER_ROW,0x41);

//Graphics area column set in Bytes !!!
 WriteData2(BYTES_PER_ROW,0x43);

//set display mode = OR
 WriteCommand(0x80);

//Text and graphic display !!
 WriteCommand(0x9C);

 ClearScreen();
}

//##########################################################
// Zeichen von Windows nach T6963 konvertieren
unsigned char ConvertText(unsigned char ch)
{
 unsigned char tmp;

 tmp=ch;

//Die case-Werte sind bezogen auf den SystemFont !!!
 switch(ch)
  {
   case 196: tmp=0x6E; break; // 
   case 228: tmp=0x64; break; // 
   case 214: tmp=0x79; break; // 
   case 246: tmp=0x74; break; // 
   case 220: tmp=0x7A; break; // 
   case 252: tmp=0x61; break; // 
   case 223: tmp=223; break; // 
   default: tmp-=0x20; break;
  }

 return tmp;
}

//##########################################################
// Text horizontal schreiben
void LCDText(char *txt, unsigned char xpos, unsigned char ypos)
{
 char ps;

//Zeiger auf neue Textposition
 WriteData2(xpos+(BYTES_PER_ROW)*ypos,0x24);

 while((ps = *txt)) //Solange Zeichen grer 0x00
  {
   txt++; //Nchstes Zeichen
   if (ps== 0) break; //Raus wenn Zeichen gleich 0
   ps=ConvertText(ps); //bersetzungstabelle aufrufen
   WriteData1(ps,0xC0); //Write ps, Auto Inkrement
  }
}

//##########################################################
// Kompletten TextBildschirm lschen
void ClearTextScreen()
//##########################################################
{
 int i;

//Write text home address
//Set address pointer = Text home
 WriteData2(T_BASE,0x24);

//ClearText RAM
 for(i=0; i<(BYTES_PER_ROW * LCD_HEIGHT/8); i++) WriteData1(0x00,0xC0);
}

//##########################################################
// Lsche Bereich im Grafikschirm
void ClearGScreen(unsigned int begin, unsigned int end)
//##########################################################
{
 unsigned int i;

 WriteData2(begin,0x24); //Setze Startadresse des zu lschenden Bereiches

 for(i=begin; i<=end; i++) WriteData1(0x00,0xC0);
}

//##########################################################
//Lscht den TextBildschirm, den Grafikschirm und setzt den
//Grafikcursor auf 0,0
void ClearScreen()
//##########################################################
{
 ClearGScreen(G_BASE,G_END1);
 ClearTextScreen();
 SetGCursor(0,0);
}

//##########################################################
// Set graphic screen base line in virtual screen
void SetGBaseLine(unsigned char line)
//##########################################################
{
 unsigned int adr;
 
 adr=G_BASE;
 adr+=BYTES_PER_ROW * line;

 WriteData2(adr,0x42); //set new startadress of screen
}
