//######################################################################
// File: buttons.c
//
// Check the player buttons.
//
//######################################################################
// Last change: 05.05.2007
//#########################################################################
// hk@holger-klabunde.de
// http://www.holger-klabunde.de/index.html
//#########################################################################
// Compiler: AVR-GCC 3.4.6
//#########################################################################
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>

#include "mydefs.h"
#include "../dos.h"

#include "buttons.h"
#include "vs1001.h"
#include "display.h"
#include "rc5.h"

#ifdef USE_PLAYER_BUTTONS
 volatile unsigned char button_pressed;
 volatile unsigned char vol_button;
#endif

volatile unsigned char timeout; // max. 25.5s timeouts for 100ms timer ticks
unsigned char max_hangs;

//###################################################################################
void ButtonsInit(void)
//###################################################################################
{
#ifdef USE_PLAYER_BUTTONS
 PLAYER_BUTTONS_DDR=0; 	//Alles Eingnge
 PLAYER_BUTTONS_PORT=0xFF; 	//Eingnge mit Pullups setzen
#endif

//Timer1 Configuration for keyboard inputs and other tasks like counting
//down LCD refresh timer
// TCNT1 = 65535 - ((F_CPU / 64) / 1000 ); //Counter Reload for 1ms
 TCNT1 = 65535 - ((F_CPU / 64) / 10 ); //Counter Reload for 100ms

 TCCR1A=0; 
 TCCR1B=((1<<CS11) | (1<<CS10) ); // prescaler 1/64

#if defined (__AVR_ATmega32__) || defined (__AVR_ATmega323__) || defined (__AVR_ATmega161__)
 TIMSK |= (1<<TOIE1);   // enable timer1 int

#elif defined (__AVR_ATmega128__) || defined (__AVR_ATmega64__)
 TIMSK |= (1<<TOIE1);   // enable timer1 int

#elif defined (__AVR_ATmega168__) || defined (__AVR_ATmega88__)
 TIMSK1 |= (1<<TOIE1);   // enable timer1 int

#elif defined (__AVR_ATmega8__)
 TIMSK |= (1<<TOIE1);   // enable timer1 int

#else
#  error "processor type not defined in buttons.c ButtonsInit()"
#endif


#ifdef USE_PLAYER_BUTTONS
 button_pressed=0;
#endif
}

//###################################################################################
// Mainly used for reading the player buttons.
// Note: Also used for IR remote control !
// Note: Also used for LCD display refresh timer !
// 
SIGNAL(SIG_OVERFLOW1)        // signal handler for timer1 overflow interrupt
//###################################################################################
{
#ifdef USE_PLAYER_BUTTONS
 unsigned char tmp;
#endif
 
// TCNT1 = 65535 - ((F_CPU / 64) / 1000 ); //Counter Reload for 1ms
 TCNT1 = 65535 - ((F_CPU / 64) / 10 ); //Counter Reload for 100ms

#ifdef USE_PLAYER_BUTTONS
 tmp=~PLAYER_BUTTONS_PIN; //read LOW active buttons on PINx
 
 switch(tmp)
  {
   case BUTTON_PLAY:	play_status=PLAYER_PLAY; 	break; //repeats. but play is play ;)
   case BUTTON_PAUSE:	play_status=PLAYER_PAUSE;	break; //repeats. but pause is pause
   case BUTTON_STOP:	play_status=PLAYER_STOP;	break; //repeats. but stop is stop
   case BUTTON_NEXT:	play_status=PLAYER_NEXT; button_pressed=1; break; //no repeat ! waits for button_pressed==0
   case BUTTON_PREV:	play_status=PLAYER_PREV; button_pressed=1; break; //no repeat ! waits for button_pressed==0
   case BUTTON_REPLAY:	play_status=PLAYER_REPLAY; button_pressed=1; break; //no repeat ! waits for button_pressed==0
   case BUTTON_VOLUP:	vol_button=PLAYER_VOLUP;	break; //repeats. next vol up in next interrupt if button keeps pressed !
   case BUTTON_VOLDOWN:	vol_button=PLAYER_VOLDOWN;	break; //repeats. next vol down in next interrupt if button keeps pressed !
   case 0x00: button_pressed=0; break; //signal for breaking waiting loops for some buttons
   default: break; //do nothing if two or more buttons are pressed at the same time 
  }
#endif

 if(update_display>0) update_display--; //don't burn the lcd ;) LCD refresh countdown 
 if(timeout>0) timeout--;               // Timeout counter for VS1001 decoder

#ifdef USE_REMOTE
 if(remote_dead>0) remote_dead--; //block some ir codes for a while
#endif
}

//###################################################################################
void CheckButtonsInLoop(void)
//###################################################################################
{
#ifdef USE_PLAYER_BUTTONS
 if(vol_button==PLAYER_VOLUP)
  {
   //don't wait til volume button up is released ! this would stop playing 
   if(volume>0)
    { volume--; VS1001_SetVolume(volume,volume);
      eeprom_write_byte(EEP_VOLUME,volume);
    }
   vol_button=0; //don't repeat until next timer interrupt !
  } 

 if(vol_button==PLAYER_VOLDOWN)
   {
    //don't wait til volume button down is released ! this would stop playing 
    if(volume<255)
     { volume++; VS1001_SetVolume(volume,volume);
      eeprom_write_byte(EEP_VOLUME,volume);
     }
    vol_button=0; //don't repeat until next timer interrupt !
   } 
#endif
}

//###################################################################################
void CheckButtonsAfterLoop(void)
//###################################################################################
{
   if(play_status==PLAYER_NEXT)
    {
#ifdef USE_PLAYER_BUTTONS
     //ugly waiting loop :( wait til button is released
     while(button_pressed) { NOP; }
#endif
     play_status=PLAYER_PLAY; //play next file
    } 

   if(play_status==PLAYER_PREV)
    {
#ifdef USE_PLAYER_BUTTONS
     //ugly waiting loop :( wait til button is released
     while(button_pressed) { NOP; }
#endif
     if(ffblk.lastposition>1) ffblk.lastposition-=2; 		//play previous file
     else if(ffblk.lastposition>0) ffblk.lastposition--;	//play first file again
     play_status=PLAYER_PLAY; //play previous file
    } 

   if(play_status==PLAYER_REPLAY)
    {
#ifdef USE_PLAYER_BUTTONS
     //ugly waiting loop :( wait til button is released
     while(button_pressed) { NOP; }
#endif
     if(ffblk.lastposition>0) ffblk.lastposition--;	//play this file again
     play_status=PLAYER_PLAY; //play this file again
    } 
}
