//######################################################################
// File: rc5.c
//
// Function:
// Decodes 13 bit RC5 remote signals.
// A green LED blinks if a remote signal is coming in.
// It also blinks on non RC5 codes ;)
//
// Uses:
// INT0 interrupt. RC5 decoding is started by a falling edge
// on the INT0 pin. But be aware ! Interrupt duration is about 25ms.
// If VS1001 buffer is full this is no problem. There are 100ms
// pauses between rc5 code transmissions. Enough time to feed the
// decoder with a lot of bytes to play.
// Never had errors playing or decoding remote control this way.
//
// Other hardware ressources used:
// TCNTO
//
//######################################################################
// Last change: 05.05.2007
//#########################################################################
// hk@holger-klabunde.de
// http://www.holger-klabunde.de/index.html
//#########################################################################
// Compiler: AVR-GCC 3.4.6
//#########################################################################
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>

#include "mydefs.h"
#include "../dos.h"

#include "buttons.h"
#include "vs1001.h"
#include "rc5.h"


#ifdef USE_REMOTE

//These variables are changed in interrupts.
volatile unsigned int rc5_code;
volatile unsigned char rc5_status;
volatile unsigned char remote_dead; //counter for not repeating some ir codes for a dead time

//#########################################################################
SIGNAL(SIG_INTERRUPT0)        // signal handler for int0 interrupt
//#########################################################################
{
 unsigned char i,shift;
 unsigned int mask;

 rc5_status=0;
 shift=0;
 mask=(unsigned int)(1<<12); //all bits come MSB first
 rc5_code=0;
 
 loop_until_bit_is_set(IR_PIN, IR_BIT);	// skip the first start bit
 // this is blocking ! but no problem. if no signal, IR detector gives
 // high on his output
 
 GREEN_ON(); //remote control signal indicator ;)

//sampling is optimal if we sample 1/4 bit before and after
//a falling/rising edge. so delay 444us before sampling.
 TCNT0=255-(IR_DELAY>>1); //set timer with 1/4 bitrate
 while(TCNT0!=0); //wait until timer overflows

//rc5 bits in this order:
//
//second startbit
//togglebit
//5 bit device code MSB first
//6 bit command code MSB first
//
//following routine takes 2 samples every 889us from IR_IN for each bit.
//result in variable shift should be 0x01 or 0x02.
//0x00 and 0x03 are errors.

 for(i=0; i<13; i++)
  {
   GREEN_OFF();

   shift=0; //kill last sampled bits
   if(bit_is_set(IR_PIN, IR_BIT)) shift|=0x01; //high sample

   TCNT0=IR_RELOAD; //set timer with sample rate
   while(TCNT0!=0); //wait until timer overflows
 
   shift<<=1; //shift first bit left
   if(bit_is_set(IR_PIN, IR_BIT)) shift|=0x01; //high sample
   shift&=0x03; //remove old sample bits

   GREEN_ON();
   TCNT0=IR_RELOAD; //set timer with sample rate
   while(TCNT0!=0); //wait until timer overflows

   if(shift!=IR_ONE && shift!=IR_ZERO) break; //error, break the loop

   if(shift==IR_ONE) rc5_code|=mask; //insert 1 bits in IR-code

   mask>>=1; //next bit
  }

 if(shift!=IR_ONE && shift!=IR_ZERO) rc5_status=0; //error
 else rc5_status=1; //success
 
 GREEN_OFF();
}

//#########################################################################
void RemoteInit(void)
//#########################################################################
{
 // Set IR input with pullup.
 sbi(IR_PORT, IR_BIT);
 cbi(IR_DDR,  IR_BIT);

 INT0_FALLING_EDGE;
 ENABLE_INT0;
 TIMER0_CONTROL_REG=TCNT0_PRESCALE;
 
 rc5_code=0;
 rc5_status=0;
 remote_dead=0;
}

//#########################################################################
void CheckRemote(void)
//#########################################################################
{
 unsigned int remote;  //actual remote control code

 if(rc5_status>0)
  {
   remote=rc5_code; //don't use rc5_code directly. It's changed in an interrupt !

   RC5_CLR_TOGGLEBIT(remote);   //it's better, toggle bit does not toggle on every RC5 remote control
//   RC5_CLR_STARTBIT2(remote);   //didn't need startbit2 til now
//       printf("IR %02d %02d\n",RC5_GET_DEVCODE(remote),RC5_GET_COMCODE(remote)); //for debugging only

   if(remote==RC5_STOP) play_status=PLAYER_STOP;
   if(remote==RC5_PLAY) play_status=PLAYER_PLAY;
   if(remote==RC5_PAUSE) play_status=PLAYER_PAUSE;
   if(remote_dead==0)
    {
     if(remote==RC5_NEXT) { play_status=PLAYER_NEXT; }
     if(remote==RC5_PREV) { play_status=PLAYER_PREV; }

     remote_dead=RC5_DEAD; // Don't repeat this command until remote_dead becomes zero !
                           // remote_dead counts down in TCNT1 interrupt
    }

   if(remote==RC5_VOLUP)
    { if(volume>0)
       { volume--; VS1001_SetVolume(volume,volume);
         eeprom_write_byte(EEP_VOLUME,volume);
       }
    }
   if(remote==RC5_VOLDOWN)
    { if(volume<255)
       { volume++; VS1001_SetVolume(volume,volume);
         eeprom_write_byte(EEP_VOLUME,volume);
       } 
    } 

   rc5_status=0; //don't repeat !

  }//if(rc5_status>0)

}

#endif //USE_REMOTE
