//#########################################################################
// File: RC5.H
//
// Definitions for RC5/RC6 decoding.
//
//#########################################################################
// hk@holger-klabunde.de
// http://www.holger-klabunde.de/index.html
//#########################################################################
// Compiler: AVR-GCC 3.4.5
//#########################################################################
// Last change: 01.09.2006
//#########################################################################

#ifndef __RC5_H
#define __RC5_H

#define USE_REMOTE

#if defined (__AVR_ATmega32__) || defined (__AVR_ATmega323__) || defined (__AVR_ATmega644__)
#define IR_PORT     PORTD
#define IR_PIN      PIND
#define IR_DDR      DDRD
#define IR_BIT      2		//INT0

#define INT0_FALLING_EDGE	MCUCR &= ~0x03
#define ENABLE_INT0		GICR |= (1 << INT0)
#define TIMER0_CONTROL_REG	TCCR0

#elif  defined (__AVR_ATmega161__) || defined (__AVR_ATmega163__)
#define IR_PORT     PORTD
#define IR_PIN      PIND
#define IR_DDR      DDRD
#define IR_BIT      2		//INT0

#define INT0_FALLING_EDGE	MCUCR &= ~0x03
#define ENABLE_INT0		GIMSK |= (1 << INT0)
#define TIMER0_CONTROL_REG	TCCR0

#elif defined (__AVR_ATmega168__) || defined (__AVR_ATmega48__) || defined (__AVR_ATmega88__)
#define IR_PORT     PORTD
#define IR_PIN      PIND
#define IR_DDR      DDRD
#define IR_BIT      2		//INT0

#define INT0_FALLING_EDGE	EICRA &= ~0x03
#define ENABLE_INT0		EIMSK |= (1 << INT0)
#define TIMER0_CONTROL_REG	TCCR0B

#elif defined (__AVR_ATmega8__)
#define IR_PORT     PORTD
#define IR_PIN      PIND
#define IR_DDR      DDRD
#define IR_BIT      2		//INT0

#define INT0_FALLING_EDGE	MCUCR &= ~0x03
#define ENABLE_INT0		GICR |= (1 << INT0)
#define TIMER0_CONTROL_REG	TCCR0

#elif defined (__AVR_ATmega128__) || defined (__AVR_ATmega64__)

#define IR_PORT     PORTD
#define IR_PIN      PIND
#define IR_DDR      DDRD
#define IR_BIT      0		//INT0

//#define INT0_FALLING_EDGE	{ EICRA &= ~0x03; EICRA |= 0x02; }
#define INT0_FALLING_EDGE	{ EICRA &= ~( (1<<ISC01) | (1<<ISC00) ); EICRA |= (1<<ISC01); }
#define ENABLE_INT0		EIMSK |= (1 << INT0)		
#define TIMER0_CONTROL_REG	TCCR0	

#else
#  error "processor type not defined in rc5.h"
#endif

//#########################################################################
//Define your RC5 remote control type here. But only ONE of them !
//#########################################################################
//#define GRUNDIG_CITY_LINE 	//Multi device remote control (TV,TAPE,VCR,CD,PA,TUNER)
#define PHILIPS_CD_850 		//CD Player remote control
//#define PHILIPS_VR_656	//VCR remote control. Some buttons for TV
//#define PHILIPS_AV_5611 	//Very old VCR remote control. No number (0,1,2,3..) buttons :(

//#########################################################################
//Prototypes for variables and functions
//#########################################################################
volatile extern unsigned int rc5_code;
volatile extern unsigned char rc5_status;
volatile extern unsigned char remote_dead; //counter for not repeating critical ir codes for this dead time

extern unsigned int remote;  //actual remote control code

extern void RemoteInit(void);
extern void CheckRemote(void);

#define TCNT0_PRESCALE	0x03	//F_CPU/64
#define RC5_DEAD	5	//Dead time for NEXT,PREV keys in 100ms steps

//#########################################################################
//RC5 Remote Control timing parameters. Depends on your CPU speed !
//Values should be near 889us.
//#########################################################################
#if F_CPU == 4000000
 #define IR_DELAY	55  //16us steps
#endif

#if F_CPU == 6000000
 #define IR_DELAY	83  //10.6us steps
#endif

#if F_CPU == 8000000
// #define IR_DELAY	111  //8us steps, ok but not optimal
 #define IR_DELAY	110  //8us steps, bit smaller for program overhead compensation
#endif

#if F_CPU == 11059200
// #define IR_DELAY	154  //5,787us steps, ok but not optimal
 #define IR_DELAY	152  //5,787us steps, bit smaller for program overhead compensation
#endif

#if F_CPU == 16000000
// #define IR_DELAY	222  //4us steps, ok but not optimal
 #define IR_DELAY	221  //4us steps, bit smaller for program overhead compensation
#endif

//#########################################################################
//RC5 biphase sample value. Depends on your CPU speed !
//#########################################################################
#define IR_RELOAD 255-IR_DELAY

//#########################################################################
//Valid values for 0 and 1 bits of biphase decoder
//0x00 and 0x03 are errors !! 
//#########################################################################
#define IR_ONE	0x02
#define IR_ZERO	0x01

//#########################################################################
//Some useful macros
//#########################################################################
#define RC5_CLR_TOGGLEBIT(a)		(a) &= ~(1<<11)
#define RC5_SET_TOGGLEBIT(a)		(a) |=  (1<<11)
#define RC5_CLR_STARTBIT2(a)		(a) &= ~(1<<12)
#define RC5_SET_STARTBIT2(a)		(a) |=  (1<<12)
#define RC5_GET_TOGGLEBIT(a)		((a) >> 11) & 0x01
#define RC5_GET_STARTBIT2(a)		((a) >> 12) & 0x01
#define RC5_GET_DEVCODE(a)		((a) >> 6) & 0x1F
#define RC5_GET_COMCODE(a)		(a) & 0x3F

//#########################################################################
//Build an RC5,RC6 code
//s=startbit, t=togglebit, d=devcode, c=commandcode
#define MAKE_RC5_CODE(s,t,d,c) 		((s)<<12) + ((t)<<11) + ((d)<<6) + (c)

#define RC5_STARTBIT2_0		0	//only RC6 can have this
#define RC5_STARTBIT2_1		1	//RC5 always 1

//#########################################################################
//RC5 devicecodes
//#########################################################################
#define RC5_DEVICE_TV1		0	//Television 1
#define RC5_DEVICE_TV2		1	//Television 2
#define RC5_DEVICE_VCR1		5	//Video Recorder 1
#define RC5_DEVICE_VCR2		6	//Video Recorder 2
#define RC5_DEVICE_PA1		16	//Amplifier1
#define RC5_DEVICE_TAPE		18	//Tape
#define RC5_DEVICE_CD		20	//CD-Player

//#########################################################################
//RC5 Commandcodes
//#########################################################################
#define RC5_COMMAND_BUTTON_0	0
#define RC5_COMMAND_BUTTON_1	1
#define RC5_COMMAND_BUTTON_2	2
#define RC5_COMMAND_BUTTON_3	3
#define RC5_COMMAND_BUTTON_4	4
#define RC5_COMMAND_BUTTON_5	5
#define RC5_COMMAND_BUTTON_6	6
#define RC5_COMMAND_BUTTON_7	7
#define RC5_COMMAND_BUTTON_8	8
#define RC5_COMMAND_BUTTON_9	9
#define RC5_COMMAND_VOLUP	16 //Volume up
#define RC5_COMMAND_VOLDOWN	17 //Volume down
#define RC5_COMMAND_CHANTITUP	32 //Channel/Title up
#define RC5_COMMAND_CHANTITDOWN	33 //Channel/Title down
#define RC5_COMMAND_STILL	41 //VCR standing picture (Standbild)
#define RC5_COMMAND_PAUSE	48
#define RC5_COMMAND_REWIND	50
#define RC5_COMMAND_FF		52 //Fast forward
#define RC5_COMMAND_PLAY	53
#define RC5_COMMAND_STOP	54
#define RC5_COMMAND_RECORD	55

//#########################################################################
#ifdef PHILIPS_AV_5611 //Build the codes
//#########################################################################
 #define RC5_PAUSE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_STILL)
 #define RC5_PREV	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_REWIND)
 #define RC5_NEXT	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_FF)
 #define RC5_PLAY	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_PLAY)
 #define RC5_STOP	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_STOP)
 #define RC5_VOLUP	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_CHANTITUP)
 #define RC5_VOLDOWN	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_CHANTITDOWN)
#endif //PHILIPS_AV_5611

//#########################################################################
#ifdef PHILIPS_VR_656 //Build the codes
//#########################################################################
 #define RC5_NUMBER_BUTTONS	 1	//Has number buttons 0..9

 #define RC5_ZERO	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_0)
 #define RC5_ONE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_1)
 #define RC5_TWO	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_2)
 #define RC5_THREE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_3)
 #define RC5_FOUR	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_4)
 #define RC5_FIVE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_5)
 #define RC5_SIX	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_6)
 #define RC5_SEVEN	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_7)
 #define RC5_EIGHT	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_8)
 #define RC5_NINE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_BUTTON_9)
 #define RC5_PAUSE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_STILL)
 #define RC5_PREV	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,37) //speed <, because PREV gives no rc5 code !
 #define RC5_NEXT	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,38) //speed >, because NEXT gives no rc5 code !
 #define RC5_PLAY	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_PLAY)
 #define RC5_STOP	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_STOP)
 #define RC5_VOLUP	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_CHANTITUP) //up/down button above cassette
 #define RC5_VOLDOWN	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_VCR1,RC5_COMMAND_CHANTITDOWN) //up/down button above cassette
// #define RC5_VOLUP	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_TV1,RC5_COMMAND_VOLUP)   //TV volume buttons to small
// #define RC5_VOLDOWN	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_TV1,RC5_COMMAND_VOLDOWN) //use the up/down button above cassette
#endif //PHILIPS_VR_656

//#########################################################################
#ifdef GRUNDIG_CITY_LINE //Build the codes
//#########################################################################
 #define RC5_NUMBER_BUTTONS	 1	//Has number buttons 0..9

 #define RC5_ZERO	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_0)
 #define RC5_ONE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_1)
 #define RC5_TWO	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_2)
 #define RC5_THREE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_3)
 #define RC5_FOUR	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_4)
 #define RC5_FIVE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_5)
 #define RC5_SIX	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_6)
 #define RC5_SEVEN	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_7)
 #define RC5_EIGHT	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_8)
 #define RC5_NINE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_BUTTON_9)
 #define RC5_PAUSE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_PAUSE)
 #define RC5_PREV	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_CHANTITDOWN)
 #define RC5_NEXT	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_CHANTITUP)
 #define RC5_PLAY	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_PLAY)
 #define RC5_STOP	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_STOP)
 #define RC5_VOLUP	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_VOLUP)
 #define RC5_VOLDOWN	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_PA1,RC5_COMMAND_VOLDOWN)
#endif //GRUNDIG_CITY_LINE

//#########################################################################
#ifdef PHILIPS_CD_850 //Build the codes
//#########################################################################
 #define RC5_NUMBER_BUTTONS	 1	//Has number buttons 0..9

 #define RC5_ZERO	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_0) //0 button
 #define RC5_ONE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_1)
 #define RC5_TWO	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_2)
 #define RC5_THREE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_3)
 #define RC5_FOUR	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_4)
 #define RC5_FIVE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_5)
 #define RC5_SIX	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_6)
 #define RC5_SEVEN	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_7)
 #define RC5_EIGHT	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_8)
 #define RC5_NINE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_BUTTON_9)
 #define RC5_PAUSE	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_PAUSE)
 #define RC5_PREV	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_CHANTITDOWN)
 #define RC5_NEXT	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_CHANTITUP)
 #define RC5_PLAY	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_PLAY)
 #define RC5_STOP	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,RC5_COMMAND_STOP)
 #define RC5_VOLUP	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,34) //Index > button
 #define RC5_VOLDOWN	MAKE_RC5_CODE(RC5_STARTBIT2_1,0,RC5_DEVICE_CD,35) //Index < button
#endif //PHILIPS_CD_850

#endif //#ifndef __RC5_H
