//#########################################################################
//
// These routines are for VS1001 clock frequency of 12.288 MHz
//
//#########################################################################
// Last change: 05.05.2007
//#########################################################################
// hk@holger-klabunde.de
// http://www.holger-klabunde.de/index.html
//#########################################################################
// Compiler: AVR-GCC 3.4.5
//#########################################################################
#include <avr/io.h>
#include <avr/interrupt.h>
#include "vs1001.h"

#include "mydefs.h"
#include "../dos.h" 	
#include "../lcd.h" 	// for Delay1us()
#include "../mmc_spi.h" // for SPI_WRITE(), SPI_WAIT()

//#######################################################
unsigned char ReadByteSPI(void)
//#######################################################
{
 SPI_WRITE(0xFF); // Dummy write to shift in data
 SPI_WAIT();      // Wait for data
 return SPDR;     // return SPI receive register
}

//#######################################################
U16 VS1001_ReadControl(U8 adress)
//#######################################################
{
 U16 result;
 
 VS_XCS_OFF;	 // XCS Low

 SPI_WRITE(VS1001_READ); SPI_WAIT();
 SPI_WRITE(adress);      SPI_WAIT();

 result = (U16)ReadByteSPI() << 8;
 result += ReadByteSPI();

 VS_XCS_ON;	 // XCS High
 Delay1us(5);	 // see VS1001 datasheet
 return result;
}

//#######################################################
void VS1001_WriteControl(U8 adress, U16 data)
//#######################################################
{
 VS_XCS_OFF;  	   // XCS Low
 SPI_WRITE(VS1001_WRITE); SPI_WAIT();
 SPI_WRITE(adress);       SPI_WAIT();

 SPI_WRITE( (U8) (data >> 8) ); SPI_WAIT();
 SPI_WRITE( (U8) data);         SPI_WAIT();

 VS_XCS_ON;	  // XCS High
 Delay1us(5);     // see VS1001 datasheet
}

//#######################################################
void VS1001_WriteData32(U8 *data)
//#######################################################
{
 U8 i;
 volatile U8 by; // why the hell did I make this volatile ?
 
 by=*data; // get first byte from memory OUTSIDE the loop for optimal SPI performance


 for(i=0; i<32; i++)
  {
	       		
   cli(); 	  // Make sure BSYNC is not high for LONGER than 8 Bits
		  // by disabling interrupts !
   VS_BSYNC_ON;	  // BSYNC High
   SPI_WRITE(by);

   // make sure BSYNC is high at least 1 bit of SPI transmission
   // these nops are for SPI speeds:
   //
   // F_CPU/2 				(1 nop)
   // F_CPU/4                      	(three nops)
   // F_CPU/16 plus SPI double speed	(seven nops)
   // Todo: other combinations possible

   NOP; 	  
   NOP;           
   NOP;           

  #if F_CPU == 16000000
   NOP; //the only one with seven nops til now          
   NOP;           
   NOP;           
   NOP;           
  #endif

   VS_BSYNC_OFF;  // BSYNC Low
   sei();         // Reenable interrupts
   data++;        // SPI module is shifting out data here, so we have some time
   by=*data;      // to get next data out of memory before waiting for end of transmission !

   SPI_WAIT();    // wait for end of transmission
  }

}

//#######################################################
void VS1001_IO_Init(void)
//#######################################################
{
    VS_RESET_ON;						  
    sbi(VS_RESET_DDR, VS_RESET_BIT);

    VS_MOSI_OFF;
    sbi(VS_MOSI_DDR, VS_MOSI_BIT);

    VS_SS_ON;
    sbi(VS_SS_DDR, VS_SS_BIT);

    VS_BSYNC_OFF;
    sbi(VS_BSYNC_DDR, VS_BSYNC_BIT);

    VS_XCS_ON;						  
    sbi(VS_XCS_DDR, VS_XCS_BIT);

    VS_SCK_OFF;						  
    sbi(VS_SCK_DDR, VS_SCK_BIT);

    /* Set DREQ input with pullup. */
    sbi(VS_DREQ_PORT, VS_DREQ_BIT);
    cbi(VS_DREQ_DDR, VS_DREQ_BIT);

    /* Set MISO input with pullup. */
    sbi(VS_MISO_PORT, VS_MISO_BIT);
    cbi(VS_MISO_DDR, VS_MISO_BIT);
}

//#######################################################
void VS1001_Reset(U8 resetmode)
//#######################################################
{

 if(resetmode == HARD_RESET)
  {
    VS_RESET_OFF; // Hard reset the decoder						  
    Delay1ms(5);          

    VS_RESET_ON;						  
    Delay1ms(200); // Wait a short time after hardware reset
   }

//##################################################################################
// Init SPI mode to no interrupts, SPI enabled, MSB first, master mode, rising clock
//
// !!! SPI clock speed <= 3.00 MHz at 12.288MHz VS1001 clock . !!!
//##################################################################################

#if F_CPU == 16000000
    SPCR= (1<<MSTR) | (1<<SPE) | (1<<SPR0); //F_CPU/16 -> 1MHz
    SPSR=0x01; //SPI double speed. F_CPU/8 -> 2MHz
#endif

#if F_CPU == 11059200
    SPCR= (1<<MSTR) | (1<<SPE); //F_CPU/4 -> 2.75MHz
    SPSR=0x00; //No SPI double speed.
#endif

#if F_CPU == 8000000
    SPCR= (1<<MSTR) | (1<<SPE); //F_CPU/4 -> 2MHz
    SPSR=0x00; //No SPI double speed.
#endif

#if F_CPU == 6000000
    SPCR= (1<<MSTR) | (1<<SPE); //F_CPU/4 -> 1.5MHz
    SPSR=0x01; //SPI double speed. F_CPU/2 -> 3MHz
#endif

#if F_CPU == 4000000
    SPCR= (1<<MSTR) | (1<<SPE); //F_CPU/4 -> 1MHz
    SPSR=0x01; //SPI double speed. F_CPU/2 -> 2MHz
#endif

 VS1001_SendNull32(2048/32); 		  // First flush the ending of low-bitrate data streams
 					  // from LAST song.

 VS1001_WriteControl(VS_MODE_REG,VS_SM_RESET); // Software reset the decoder
 Delay1us(10);                            // Wait at least 2us here
 while(VS1001_BUSY);			  // wait for DREQ. Is low for 250us @24.576MHz at this point
 
 VS1001_SendNull32(32/32);		  // write MORE than one zero byte here to the VS1001

 VS1001_WriteControl(VS_CLOCKF_REG,0x9800);     // correction for clock 12.288Mhz
 VS1001_WriteControl(VS_INT_FCTLH_REG, 0x8008); // force clock doubler.
 
//#######################################
// Maximum SPI speed for VS1001 is CLKI/4
// This is 24.576MHz / 4 = 6.144 MHz
//#######################################

#if F_CPU == 16000000
    SPCR= (1<<MSTR) | (1<<SPE); //F_CPU/4 -> 4MHz
    SPSR=0x00; //No SPI double speed. Would be 8MHz. To fast !
#endif

#if F_CPU == 11059200
    SPCR= (1<<MSTR) | (1<<SPE); //F_CPU/4 -> 2.75MHz
    SPSR=0x01; //SPI double speed. F_CPU/2 -> 5.5MHz
#endif

#if F_CPU == 8000000
    SPCR= (1<<MSTR) | (1<<SPE); //F_CPU/4 -> 2MHz
    SPSR=0x01; //SPI double speed. F_CPU/2 -> 4MHz
#endif

#if F_CPU == 6000000
    SPCR= (1<<MSTR) | (1<<SPE); //F_CPU/4 -> 1.5MHz
    SPSR=0x01; //SPI double speed. F_CPU/2 -> 3MHz
#endif

#if F_CPU == 4000000
    SPCR= (1<<MSTR) | (1<<SPE); //F_CPU/4 -> 1MHz
    SPSR=0x01; //SPI double speed. F_CPU/2 -> 2MHz
#endif

// !!!! Important Note !!!!, MMC/SD cards would run now with new SPI speed !
// Todo: Fix this by switching SPI speed to maximum at beginning of MMC routines
// and restore old SPI speed at the end of MMC routines

// Sorry in german only ;)
//
// Bevor "force clock doubler" soll die SPI Geschwindigkeit auf 3MHz oder weniger
// eingestellt werden !? Funktioniert auch anders.
//
// Beweis: Ausgehend von einem Hardwarereset
// =========================================
// Bei 16MHz schalte ich die SPI Geschwindigkeit auf F_CPU/4, also 4MHz.
// Funktioniert bestens. Noch schlimmer bei 11MHz. Da schalte ich auch noch auf
// SPI DoubleSpeed. Also 5.5MHz. Das SPI Modul vom VS1001 luft auch damit vor
// dem "force clock doubler" Trick bei 12.288MHz VS1001 Takt.
// Es sieht so aus als knnte man das SPI Modul vom VS1001 bis CLKI/2
// takten ohne Probleme zu bekommen. Oder ist der Chip den ich bekommen habe
// einfach nur schneller als sonst ?
 
}

//#######################################################
void VS1001_SetVolume(U8 left,U8 right)
//#######################################################
{
 VS1001_WriteControl(VS_VOL_REG,((U16)left << 8) | right );
}

//#######################################################
// Values for mode:
//
// VS_SM_DIFF Left channel inverted.
// VS_SM_FFWD Fast forward.
// VS_SM_RESET Software reset.
// VS_SM_PDOWN Switch to power down mode.
// VS_SM_BASS Bass/treble enhancer.
//#######################################################
void VS1001_SetMode(U16 mode)
{ VS1001_WriteControl(VS_MODE_REG, mode); }

//#######################################################
// Sends count*32 zero bytes to VS1001
void VS1001_SendNull32(U8 count)
//#######################################################
{
 U8 i, buf[32], *p;
 
// for(i=0; i<32; i++) buf[i]=0;
 p = buf;
 i = 32;
 while(i)
  {
   *p++ = 0;
   i--;
  } 
  
 i = count;
 while(i)
  {
   while(VS1001_BUSY);  // wait for DREQ.
   VS1001_WriteData32(buf);
   i--;
  } 
}

//#######################################################
U16 VS1001_GetPlayTime(void)
//#######################################################
{ return VS1001_ReadControl(VS_DECODE_TIME_REG); }

//#######################################################
U16 VS1001_GetAudata(void)
//#######################################################
{ return VS1001_ReadControl(VS_AUDATA_REG); }

//#######################################################
U16 VS1001_GetHdat0(void)
//#######################################################
{ return VS1001_ReadControl(VS_HDAT0_REG); }


